import grpc
from concurrent import futures
import logging

import calculator_pb2
import calculator_pb2_grpc

logging.basicConfig(level=logging.INFO, format='%(asctime)s [%(levelname)s] %(message)s')
logger = logging.getLogger(__name__)


class CalculatorServicer(calculator_pb2_grpc.CalculatorServicer):
    """Implementation of the Calculator service"""
    
    def Add(self, request, context):
        """Add two numbers"""
        logger.info(f"Add({request.a}, {request.b})")
        result = request.a + request.b
        return calculator_pb2.Result(value=result)
    
    def Subtract(self, request, context):
        """Subtract b from a"""
        logger.info(f"Subtract({request.a}, {request.b})")
        result = request.a - request.b
        return calculator_pb2.Result(value=result)
    
    def Multiply(self, request, context):
        """Multiply two numbers"""
        logger.info(f"Multiply({request.a}, {request.b})")
        result = request.a * request.b
        return calculator_pb2.Result(value=result)
    
    def Divide(self, request, context):
        """Divide a by b"""
        logger.info(f"Divide({request.a}, {request.b})")
        if request.b == 0:
            context.set_code(grpc.StatusCode.INVALID_ARGUMENT)
            context.set_details("Cannot divide by zero")
            return calculator_pb2.Result()
        result = request.a / request.b
        return calculator_pb2.Result(value=result)


def serve():
    server = grpc.server(futures.ThreadPoolExecutor(max_workers=10))
    calculator_pb2_grpc.add_CalculatorServicer_to_server(
        CalculatorServicer(), server
    )
    
    port = '50051'
    server.add_insecure_port(f'[::]:{port}')
    logger.info(f"gRPC server listening on port {port}")
    
    try:
        server.start()
        server.wait_for_termination()
    except KeyboardInterrupt:
        logger.info("Shutting down server")
        server.stop(0)


if __name__ == '__main__':
    serve()

