from xmlrpc.server import SimpleXMLRPCServer
import logging

# Configure logging to show timestamps and severity levels
# format: controls the output format of log messages
#   %(asctime)s - timestamp when the log was created
#   %(levelname)s - severity level (INFO, WARNING, ERROR, etc.)
#   %(message)s - the actual log message
# level: only show messages at INFO level or higher (INFO, WARNING, ERROR, CRITICAL)
logging.basicConfig(level=logging.INFO, format='%(asctime)s [%(levelname)s] %(message)s')
logger = logging.getLogger(__name__)  # Create a logger for this module


class Calculator:
    """A simple calculator providing arithmetic operations"""
    
    def add(self, a, b):
        """Add two numbers"""
        logger.info(f"add({a}, {b})")
        return a + b
    
    def subtract(self, a, b):
        """Subtract b from a"""
        logger.info(f"subtract({a}, {b})")
        return a - b
    
    def multiply(self, a, b):
        """Multiply two numbers"""
        logger.info(f"multiply({a}, {b})")
        return a * b
    
    def divide(self, a, b):
        """Divide a by b"""
        logger.info(f"divide({a}, {b})")
        if b == 0:
            logger.error("Divide by zero")
            raise ValueError("Cannot divide by zero")
        return a / b

def main():
    # Create server
    server = SimpleXMLRPCServer(('localhost', 8000), logRequests=True)
    logger.info("XML-RPC server listening on port 8000")
    
    # Register the calculator instance
    # The instance's public methods become remotely callable
    server.register_instance(Calculator())
    
    # Register a function to shut down gracefully
    server.register_function(lambda: server.shutdown(), 'shutdown')
    
    # Start serving requests
    try:
        server.serve_forever()
    except KeyboardInterrupt:
        logger.info("Shutting down server")


if __name__ == '__main__':
    main()
